/**
 * Single Property Page JavaScript
 * 
 * @package Elite_Northeast_Realty
 */

(function($) {
    'use strict';

    // Gallery Functionality
    window.changeMainImage = function(src) {
        const mainImage = document.getElementById('mainImage');
        if (mainImage) {
            mainImage.src = src;
        }
    };

    // Modal Functions
    window.openTourModal = function() {
        document.getElementById('tourModal').classList.remove('hidden');
        document.getElementById('tourModal').classList.add('flex');
        document.body.style.overflow = 'hidden';
    };

    window.closeTourModal = function() {
        document.getElementById('tourModal').classList.add('hidden');
        document.getElementById('tourModal').classList.remove('flex');
        document.body.style.overflow = '';
    };

    window.openContactModal = function() {
        document.getElementById('contactModal').classList.remove('hidden');
        document.getElementById('contactModal').classList.add('flex');
        document.body.style.overflow = 'hidden';
    };

    window.closeContactModal = function() {
        document.getElementById('contactModal').classList.add('hidden');
        document.getElementById('contactModal').classList.remove('flex');
        document.body.style.overflow = '';
    };

    window.openInfoModal = function() {
        document.getElementById('infoModal').classList.remove('hidden');
        document.getElementById('infoModal').classList.add('flex');
        document.body.style.overflow = 'hidden';
    };

    window.closeInfoModal = function() {
        document.getElementById('infoModal').classList.add('hidden');
        document.getElementById('infoModal').classList.remove('flex');
        document.body.style.overflow = '';
    };

    // Scroll to calculator
    window.scrollToCalculator = function() {
        const calculator = document.getElementById('pre-qualification-form');
        if (calculator) {
            calculator.scrollIntoView({ behavior: 'smooth', block: 'center' });
        }
    };

    // Close modals on backdrop click
    $(document).on('click', '#tourModal, #contactModal, #infoModal', function(e) {
        if (e.target === this) {
            if (this.id === 'tourModal') closeTourModal();
            if (this.id === 'contactModal') closeContactModal();
            if (this.id === 'infoModal') closeInfoModal();
        }
    });

    // Close modals on ESC key
    $(document).on('keydown', function(e) {
        if (e.key === 'Escape') {
            closeTourModal();
            closeContactModal();
            closeInfoModal();
        }
    });

    // Credit Score Range Update
    $(document).on('input', 'input[name="credit_score"]', function() {
        const value = $(this).val();
        $('#credit-score-value').text(value);
        
        // Update gradient background
        const min = $(this).attr('min');
        const max = $(this).attr('max');
        const percentage = ((value - min) / (max - min)) * 100;
        $(this).css('background', `linear-gradient(to right, #2563eb 0%, #2563eb ${percentage}%, #e5e7eb ${percentage}%, #e5e7eb 100%)`);
    });

    // Initialize credit score display
    $(document).ready(function() {
        const creditScore = $('input[name="credit_score"]');
        if (creditScore.length) {
            creditScore.trigger('input');
        }
    });

    // Pre-qualification Calculator
    $('#pre-qualification-form').on('submit', function(e) {
        e.preventDefault();
        
        const form = $(this);
        const submitBtn = form.find('button[type="submit"]');
        const resultDiv = $('#qualification-result');
        
        // Add loading state
        submitBtn.addClass('loading').prop('disabled', true);
        
        // Get form data
        const income = parseFloat(form.find('input[name="income"]').val().replace(/[^0-9.]/g, ''));
        const downPayment = parseFloat(form.find('input[name="down_payment"]').val().replace(/[^0-9.]/g, ''));
        const creditScore = parseInt(form.find('input[name="credit_score"]').val());
        
        // Simple qualification calculation
        const monthlyIncome = income / 12;
        const maxMonthlyPayment = monthlyIncome * 0.28; // 28% DTI ratio
        const loanAmount = (maxMonthlyPayment * 360) / 2; // Simplified calculation
        const totalBudget = loanAmount + downPayment;
        
        // Simulate API call delay
        setTimeout(function() {
            // Remove loading state
            submitBtn.removeClass('loading').prop('disabled', false);
            
            // Show result
            let resultHTML = '';
            if (creditScore >= 620 && income > 0) {
                resultHTML = `
                    <div class="bg-green-50 border border-green-200 rounded-lg p-4">
                        <h4 class="font-semibold text-green-900 mb-2">You may qualify for up to:</h4>
                        <p class="text-2xl font-bold text-green-900">$${totalBudget.toLocaleString()}</p>
                        <p class="text-sm text-green-700 mt-2">Based on your income and down payment</p>
                        <button onclick="openContactModal()" class="mt-3 w-full bg-green-600 text-white py-2 rounded-lg hover:bg-green-700 transition-colors font-semibold">
                            Get Official Pre-Approval
                        </button>
                    </div>
                `;
            } else {
                resultHTML = `
                    <div class="bg-yellow-50 border border-yellow-200 rounded-lg p-4">
                        <h4 class="font-semibold text-yellow-900 mb-2">Let's discuss your options</h4>
                        <p class="text-sm text-yellow-700">Based on your information, we recommend speaking with one of our mortgage specialists.</p>
                        <button onclick="openContactModal()" class="mt-3 w-full bg-yellow-600 text-white py-2 rounded-lg hover:bg-yellow-700 transition-colors font-semibold">
                            Speak with a Specialist
                        </button>
                    </div>
                `;
            }
            
            resultDiv.html(resultHTML).removeClass('hidden');
        }, 1000);
    });

    // Form Submissions via AJAX
    function handleFormSubmission(formId, successMessage) {
        $(formId).on('submit', function(e) {
            e.preventDefault();
            
            const form = $(this);
            const submitBtn = form.find('button[type="submit"]');
            const formData = form.serialize();
            
            // Add loading state
            submitBtn.addClass('loading').prop('disabled', true);
            
            // Submit via AJAX
            $.ajax({
                url: propertyAjax.ajaxurl,
                type: 'POST',
                data: {
                    action: 'submit_property_form',
                    nonce: propertyAjax.nonce,
                    form_type: formId.replace('#', '').replace('-form', ''),
                    form_data: formData
                },
                success: function(response) {
                    if (response.success) {
                        // Replace form with success message
                        form.replaceWith(`
                            <div class="bg-green-50 border border-green-200 rounded-lg p-6 text-center">
                                <div class="w-16 h-16 bg-green-100 rounded-full flex items-center justify-center mx-auto mb-4">
                                    <i class="ri-check-line text-green-600 text-2xl"></i>
                                </div>
                                <h4 class="font-semibold text-green-900 mb-2">${successMessage}</h4>
                                <p class="text-sm text-green-700">We'll be in touch within 24 hours.</p>
                            </div>
                        `);
                        
                        // Close modal after 3 seconds
                        setTimeout(function() {
                            if (formId === '#tour-request-form') closeTourModal();
                            if (formId === '#contact-agent-form') closeContactModal();
                            if (formId === '#request-info-form') closeInfoModal();
                        }, 3000);
                    } else {
                        // Show error message
                        const errorMsg = $('<div class="bg-red-50 border border-red-200 rounded-lg p-3 mb-3 text-sm text-red-700">').text(response.data || 'An error occurred. Please try again.');
                        form.prepend(errorMsg);
                        
                        // Remove error after 5 seconds
                        setTimeout(function() {
                            errorMsg.fadeOut(function() {
                                $(this).remove();
                            });
                        }, 5000);
                    }
                },
                error: function() {
                    // Show generic error
                    const errorMsg = $('<div class="bg-red-50 border border-red-200 rounded-lg p-3 mb-3 text-sm text-red-700">').text('Connection error. Please try again later.');
                    form.prepend(errorMsg);
                },
                complete: function() {
                    // Remove loading state
                    submitBtn.removeClass('loading').prop('disabled', false);
                }
            });
        });
    }

    // Initialize form handlers
    $(document).ready(function() {
        handleFormSubmission('#tour-request-form', 'Tour Request Sent!');
        handleFormSubmission('#contact-agent-form', 'Message Sent!');
        handleFormSubmission('#request-info-form', 'Information Request Sent!');
    });

    // Initialize map if Google Maps is available
    window.initPropertyMap = function() {
        const mapElement = document.getElementById('property-map');
        if (mapElement && typeof google !== 'undefined') {
            // Get property coordinates from data attributes
            const lat = parseFloat(mapElement.dataset.lat) || 40.7128;
            const lng = parseFloat(mapElement.dataset.lng) || -74.0060;
            
            const map = new google.maps.Map(mapElement, {
                center: { lat: lat, lng: lng },
                zoom: 15,
                styles: [
                    {
                        featureType: 'poi',
                        elementType: 'labels',
                        stylers: [{ visibility: 'off' }]
                    }
                ]
            });
            
            const marker = new google.maps.Marker({
                position: { lat: lat, lng: lng },
                map: map,
                title: 'Property Location'
            });
        }
    };

    // Tab functionality for future expansion
    $(document).on('click', '.tab-button', function() {
        const $this = $(this);
        const target = $this.data('target');
        
        // Update active states
        $('.tab-button').removeClass('active');
        $this.addClass('active');
        
        // Show/hide content
        $('.tab-content').addClass('hidden');
        $(target).removeClass('hidden');
    });

    // Lightbox for gallery images (future enhancement)
    $(document).on('click', '.gallery-thumb img', function() {
        const src = $(this).attr('src');
        // Could implement a full lightbox here
        changeMainImage(src);
    });

    // Smooth scroll for anchor links
    $(document).on('click', 'a[href^="#"]', function(e) {
        const target = $(this.getAttribute('href'));
        if (target.length) {
            e.preventDefault();
            $('html, body').animate({
                scrollTop: target.offset().top - 100
            }, 500);
        }
    });

    // Format currency inputs
    $(document).on('blur', 'input[name="income"], input[name="down_payment"]', function() {
        const value = $(this).val().replace(/[^0-9.]/g, '');
        if (value) {
            $(this).val('$' + parseFloat(value).toLocaleString());
        }
    });

    // Share functionality
    window.shareProperty = function(platform) {
        const url = encodeURIComponent(window.location.href);
        const title = encodeURIComponent(document.title);
        let shareUrl = '';
        
        switch(platform) {
            case 'facebook':
                shareUrl = `https://www.facebook.com/sharer/sharer.php?u=${url}`;
                break;
            case 'twitter':
                shareUrl = `https://twitter.com/intent/tweet?url=${url}&text=${title}`;
                break;
            case 'email':
                shareUrl = `mailto:?subject=${title}&body=Check out this property: ${url}`;
                break;
        }
        
        if (shareUrl) {
            window.open(shareUrl, '_blank', 'width=600,height=400');
        }
    };

})(jQuery);